package furny.states;

import furny.ga.FurnLayoutIndividual;
import ga.core.GA;
import ga.core.algorithm.interactive.ISIGA;
import ga.core.evaluation.IInteractiveFitnessEvaluator;
import ga.view.appstate.SceneState;
import ga.view.appstate.menu.MenuListener;
import ga.view.factory.EffectsFactory;
import ga.view.input.CamDragListener;
import ga.view.interfaces.IPhenotypeGenerator;
import ga.view.interfaces.IPreEvaluationState;
import ga.view.streaming.showroom.BoxShowRoom.LightingType;
import ga.view.streaming.showroom.CameraSettings;
import ga.view.streaming.showroom.ShowRoom;
import ga.view.streaming.showroom.ShowRoomFactory;
import ga.view.streaming.showroom.ShowRoomSettings;
import ga.view.streaming.showroom.ShowRoomSettings.ShowRoomType;

import java.util.concurrent.Callable;
import java.util.logging.Logger;

import com.jme3.app.Application;
import com.jme3.app.state.AppStateManager;
import com.jme3.input.KeyInput;
import com.jme3.input.controls.ActionListener;
import com.jme3.input.controls.KeyTrigger;
import com.jme3.material.Material;
import com.jme3.math.ColorRGBA;
import com.jme3.math.Vector3f;
import com.jme3.scene.Node;
import com.jme3.system.AppSettings;

/**
 * Pre state to initialize the population and show room.
 * 
 * @since 12.08.2012
 * @author Stephan Dreyer
 */
public class InitializerPreState extends SceneState implements
    IPreEvaluationState {

  // the logger for this class
  private static final Logger LOGGER = Logger
      .getLogger(InitializerPreState.class.getName());

  private final IInteractiveFitnessEvaluator<FurnLayoutIndividual> evaluator;
  private final ISIGA<FurnLayoutIndividual> algorithm;

  private ShowRoomFactory showRoomFactory;
  private ShowRoom showRoom;
  private final ShowRoomSettings srSettings = new ShowRoomSettings();
  private CameraSettings camSettings;

  private final IPhenotypeGenerator<FurnLayoutIndividual, Node> phenotypeGenerator;

  private CamDragListener dragListener;

  private MenuListener menuListener;

  private AppSettings settings;

  private InitializerUIState uiState;
  private FurnLayoutIndividual ind;

  /**
   * Instantiates a new initializer pre state.
   * 
   * @param evaluator
   *          the evaluator
   * @param algorithm
   *          the algorithm
   * @param phenotypeGenerator
   *          the phenotype generator
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  public InitializerPreState(
      final IInteractiveFitnessEvaluator<FurnLayoutIndividual> evaluator,
      final ISIGA<FurnLayoutIndividual> algorithm,
      final IPhenotypeGenerator<FurnLayoutIndividual, Node> phenotypeGenerator) {
    this.evaluator = evaluator;
    this.algorithm = algorithm;
    this.phenotypeGenerator = phenotypeGenerator;
  }

  @Override
  public void setMenuListener(final MenuListener menuListener) {
    this.menuListener = menuListener;
  }

  @Override
  public void initialize(final AppStateManager stateManager,
      final Application app) {
    super.initialize(stateManager, app);

    this.settings = app.getContext().getSettings();

    srSettings.put(ShowRoomSettings.TYPE, ShowRoomType.BOX);
    srSettings.put(ShowRoomSettings.BOX_WIDTH, 5f);
    srSettings.put(ShowRoomSettings.BOX_LENGTH, 5f);
    srSettings.put(ShowRoomSettings.BOX_HEIGHT, 2.6f);

    showRoomFactory = new ShowRoomFactory(assetManager, settings, srSettings);
    updateShowRoom();

    camSettings = CameraSettings.getOrthographicSettings(showRoom, settings);
    camSettings.configureCamera(cam);

    dragListener = new CamDragListener(cam, inputManager, camSettings);

    inputManager.addMapping("switch view", new KeyTrigger(KeyInput.KEY_V));
    inputManager.addListener(new ViewSwitchListener(), "switch view");

    // this seems to cause problems
    // EffectsFactory.addShadowProcessor(assetManager, settings, viewPort,
    // new Vector3f(3.0f, 2.1f, 3.0f));
    EffectsFactory.addLightScatteringProcessor(assetManager, inputManager,
        settings, viewPort, new Vector3f(3.0f, 1.9f, 3.0f));
    EffectsFactory.addSSAOProcessor(assetManager, inputManager, settings,
        viewPort);

    uiState = new InitializerUIState(evaluator, this, menuListener);
    stateManager.attach(uiState);
  }

  /**
   * Sets the floor texture.
   * 
   * @param name
   *          the new floor texture
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  public void setFloorTexture(final String name) {
    srSettings.put(ShowRoomSettings.FLOOR_MATERIAL,
        assetManager.loadMaterial(name));

    updateShowRoom();
  }

  /**
   * Sets the wall color.
   * 
   * @param color
   *          the new wall color
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  public void setWallColor(final ColorRGBA color) {
    app.enqueue(new Callable<Void>() {
      @Override
      public Void call() throws Exception {
        uiState.updateView();

        final Material mat = showRoom.getWallMaterial().clone();
        mat.setColor("Diffuse", color);
        mat.setColor("Ambient", color.mult(0.8f));
        srSettings.put(ShowRoomSettings.WALL_MATERIAL, mat);

        updateShowRoom();

        return null;
      }
    });
  }

  /**
   * Saves settings in the GA context.
   * 
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  public void saveSettings() {
    algorithm.getContext().put(GA.KEY_VALIDATION_SPACE, showRoom);

    if (ind != null) {
      algorithm.getContext().put(GA.KEY_INIT_INDIVIDUAL, ind);
    }
  }

  /**
   * Sets the population percentage that should be initialized with the loaded
   * individual.
   * 
   * @param percentage
   *          the new pop percentage
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  public void setPopPercentage(final int percentage) {
    app.enqueue(new Callable<Void>() {
      @Override
      public Void call() throws Exception {
        algorithm.getContext().put(GA.KEY_INIT_INDIVIDUAL_PERCENTAGE,
            percentage);

        uiState.updateView();

        return null;
      }
    });
  }

  /**
   * Gets the population percentage that should be initialized with the loaded
   * individual as string.
   * 
   * @return the pop percentage string
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  public String getPopPercentageString() {
    return getPopPercentage() + "% of Population";
  }

  /**
   * Gets the population percentage that should be initialized with the loaded
   * individual.
   * 
   * @return the pop percentage
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  public int getPopPercentage() {
    return algorithm.getContext().getInt(GA.KEY_INIT_INDIVIDUAL_PERCENTAGE, 5);
  }

  /**
   * Gets the room size as string.
   * 
   * @return the room size string
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  public String getRoomSizeString() {
    return getRoomWidth() + "cm * " + getRoomLength() + "cm * "
        + getRoomHeight() + "cm";
  }

  /**
   * Gets the room width.
   * 
   * @return the room width
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  public int getRoomWidth() {
    return (int) (srSettings.getFloat(ShowRoomSettings.BOX_WIDTH, 5f) * 100f);
  }

  /**
   * Gets the room length.
   * 
   * @return the room length
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  public int getRoomLength() {
    return (int) (srSettings.getFloat(ShowRoomSettings.BOX_LENGTH, 5f) * 100f);
  }

  /**
   * Gets the room height.
   * 
   * @return the room height
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  public int getRoomHeight() {
    return (int) (srSettings.getFloat(ShowRoomSettings.BOX_HEIGHT, 2.6f) * 100f);
  }

  /**
   * Sets the room size.
   * 
   * @param width
   *          the width
   * @param length
   *          the length
   * @param height
   *          the height
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  public void setRoomSize(final int width, final int length, final int height) {
    app.enqueue(new Callable<Void>() {
      @Override
      public Void call() throws Exception {
        srSettings.put(ShowRoomSettings.BOX_WIDTH, width / 100f);
        srSettings.put(ShowRoomSettings.BOX_LENGTH, length / 100f);
        srSettings.put(ShowRoomSettings.BOX_HEIGHT, height / 100f);

        uiState.updateView();

        updateShowRoom();

        return null;
      }
    });
  }

  /**
   * Sets a new individual.
   * 
   * @param ind
   *          the new individual
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  public void setIndividual(final FurnLayoutIndividual ind) {
    app.enqueue(new Callable<Void>() {
      @Override
      public Void call() throws Exception {
        InitializerPreState.this.ind = ind;

        uiState.updateView();

        updateShowRoom();

        return null;
      }
    });
  }

  /**
   * Gets the current individual.
   * 
   * @return the individual
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  public FurnLayoutIndividual getIndividual() {
    return ind;
  }

  /**
   * Sets the lighting type.
   * 
   * @param lightingType
   *          the new lighting type
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  public void setLightingType(final LightingType lightingType) {
    app.enqueue(new Callable<Void>() {
      @Override
      public Void call() throws Exception {
        srSettings.put(ShowRoomSettings.LIGHTING_TYPE, lightingType);

        updateShowRoom();

        return null;
      }
    });
  }

  /**
   * Updates show room.
   * 
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  public void updateShowRoom() {
    rootNode.detachAllChildren();

    showRoom = showRoomFactory.createShowRoom(true);

    if (ind != null) {
      try {
        showRoom.setPhenotype(phenotypeGenerator.createPhenotype(ind));
      } catch (final Exception e) {
        LOGGER.warning("Error creating phenotype");
      }
    }

    rootNode.attachChild(showRoom);
  }

  @Override
  public void setEnabled(final boolean enabled) {
    super.setEnabled(enabled);

    // if (viewPort != null) {
    // if (enabled) {
    // if (!renderManager.getMainViews().contains(viewPort)) {
    // viewPort = renderManager.createMainView("Scene", cam);
    // viewPort.setClearFlags(true, true, true);
    // viewPort.attachScene(rootNode);
    // }
    // } else {
    // // DO NOT REMOVE PROCESSORS HERE
    // // while (viewPort.getProcessors().size() > 0) {
    // // SceneProcessor proc = viewPort.getProcessors().get(0);
    // // viewPort.removeProcessor(proc);
    // // }
    //
    // renderManager.removeMainView(viewPort);
    // }
    // }

    if (dragListener != null) {
      dragListener.setEnabled(enabled);
    }

    if (uiState != null) {
      uiState.setEnabled(enabled);
    }
  }

  @Override
  public void cleanup() {
    super.cleanup();

    stateManager.detach(uiState);
  }

  // private class MenuNode extends Node {
  // private Vector3f targetLocation;
  // private Vector3f orgLocation;
  //
  // private boolean selected;
  // private final String name;
  //
  // private final BitmapText text;
  //
  // public MenuNode(final T ind) {
  // this.name = ind.getId() + "";
  // setName(name + " node");
  //
  // final Material whitemat = new Material(assetManager,
  // "Common/MatDefs/Misc/Unshaded.j3md");
  // whitemat.setColor("Color", ColorRGBA.White);
  //
  // final float boxSizeX = .8f;
  // final float boxSizeY = .6f;
  // final float borderSize = .05f;
  //
  // final Material mat = new Material(assetManager,
  // "Common/MatDefs/Misc/Unshaded.j3md");
  // mat.setTexture("ColorMap",
  // assetManager.loadTexture(gaSetting.get("iconImage", "")));
  //
  // final Geometry geo = new Geometry("Menu node", new Box(boxSizeY, 0.01f,
  // boxSizeX));
  // geo.setMaterial(mat);
  // geo.setShadowMode(ShadowMode.Off);
  //
  // final Box box = new Box(boxSizeY + borderSize, 0f, boxSizeX + borderSize);
  // final Geometry outterBox = new Geometry(name + " outter box", box);
  // outterBox.setMaterial(whitemat);
  //
  // attachChild(outterBox);
  //
  // attachChild(geo);
  //
  // final BitmapFont font = assetManager
  // .loadFont("Interface/Fonts/Default.fnt");
  //
  // text = new BitmapText(font, false);
  // text.setName(name + " text");
  // text.setSize(0.1f);
  // text.setText(name);
  // text.setCullHint(CullHint.Never);
  //
  // text.setBox(new Rectangle(-1f, 0f, 2f, text.getLineHeight()));
  // text.setAlignment(Align.Center);
  //
  // text.setLocalTranslation(boxSizeY + borderSize, 0f, 0f);
  //
  // rotate(0f, FastMath.DEG_TO_RAD * 90f, 0f);
  //
  // attachChild(text);
  // text.rotate(FastMath.DEG_TO_RAD * 270f, FastMath.DEG_TO_RAD * 90f, 0);
  //
  // }
  //
  // public void setOrgLocation(final Vector3f orgLocation) {
  // this.orgLocation = orgLocation;
  // setLocalTranslation(orgLocation);
  // }
  //
  // @Override
  // public void updateLogicalState(final float tpf) {
  // super.updateLogicalState(tpf);
  //
  // if (targetLocation != null) {
  // final Vector3f loc = getLocalTranslation();
  //
  // final float distance = targetLocation.distance(loc);
  //
  // if (distance > .05f) {
  // // the move step should be limited by the distance to the target
  // final float maxMoveStep = Math.min(distance, tpf * 3f);
  //
  // // calc the delta translation by subtracting the direction vector
  // // multiplied with the step size from the target location
  // final Vector3f dir = targetLocation.subtract(loc).normalize()
  // .mult(maxMoveStep);
  // setLocalTranslation(loc.add(dir));
  // }
  // }
  // }
  //
  // public void setSelected(final boolean select) {
  // if (select != selected) {
  // selected = select;
  // LOGGER.warning(this + " is selected: " + select);
  // if (select) {
  // this.targetLocation = orgLocation.mult(.9f).add(0f, .5f, 0f);
  // } else {
  // this.targetLocation = this.orgLocation;
  // }
  // } else if (selected) {
  // doubleClickListener.setEnabled(false);
  // listener.modelSelected(name);
  //
  // // / reset
  // this.targetLocation = this.orgLocation;
  // selected = false;
  // }
  // }
  // }

  /**
   * Keyboard listener to switch the perspective.
   * 
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  private class ViewSwitchListener implements ActionListener {
    private int i = 1;

    @Override
    public void onAction(final String name, final boolean keyPressed,
        final float tpf) {
      if (!keyPressed) {
        final int length = CameraSettings.Type.values().length;

        final CameraSettings.Type type = CameraSettings.Type.values()[++i
            % length];

        camSettings = CameraSettings.getSettings(type,
            showRoomFactory.createShowRoom(), settings);

        camSettings.configureCamera(cam);
        dragListener.init(camSettings);

        LOGGER.info("Camera is now " + type);
      }
    }
  }
}
